//////////////////////////////////////////////
//
// RatingMgrDBUtils.java
//
 

package Alkindi.Services.ServicesImpl;

import java.util.*;
import java.sql.*;
import java.math.*;
import oracle.jdbc.driver.*;
import Alkindi.Services.*;
import Alkindi.Data.*;
import Alkindi.Services.Util.*;
import Alkindi.Services.InternalData.*;

/* 
$Header: RatingMgrDBUtils.java, 36, 5/11/01 11:07:04 AM, Schwartz, Joe$
$Log: 
 36   Alkindi Development1.35        5/11/01 11:07:04 AM  Schwartz, Joe   Added
      body to method getUserEvalStats(), which now populates an array of
      UserEvalStats objects with data from two stored procs. 
 35   Alkindi Development1.34        5/8/01 10:08:21 AM   Schwartz, Joe  
      Changed references to UserEvalInfo to new UserEvalStats class in
      InternalData package. Changed method RatMgrDBUtils.getUserEvalInfo() to
      getUserEvalStats().
 34   Alkindi Development1.33        5/8/01 9:57:34 AM    Schwartz, Joe   Added
      new nested class UserEvalInfo and new method getUserEvalInfo() to deal
      with stats necesssary for selecting products to rate. Changed use of
      InternalEJBManager to InternalComponentManager. Removed dependencies on
      EJB from these classes. Cleaned up comments.
 33   Alkindi Development1.32        4/26/01 2:37:16 PM   Schwartz, Joe  
      Modifed to account for new InternalData package.
 32   Alkindi Development1.31        4/26/01 1:11:14 PM   Schwartz, Joe   Moved
      into new ServicesImpl package.
 31   Alkindi Development1.30        3/14/01 5:38:47 PM   Schwartz, Joe  
      Regenerated from Rose.
 30   Alkindi Development1.29        2/23/01 4:33:25 PM   Schwartz, Joe   Small
      fixes accomodating new integer product ID.
 29   Alkindi Development1.28        2/13/01 5:55:31 PM   Schwartz, Joe  
      Changed to account for new Product id int type and SparseRatingsArray.
 28   Alkindi Development1.27        2/5/01 3:08:22 PM    Schwartz, Joe  
      Changed to accomodate new AlkConn members.
 27   Alkindi Development1.26        1/25/01 9:58:07 PM   Schwartz, Joe  
      Working on speed, sensibility.
 26   Alkindi Development1.25        12/29/00 1:14:37 PM  Schwartz, Joe  
      Removed getNumProdSeen. Moved that method into UserStat.
 25   Alkindi Development1.24        12/28/00 1:10:19 PM  Schwartz, Joe   Added
      Version Control header info.
      Changed package-scope members to protected.
 24   Alkindi Development1.23        12/28/00 12:05:51 PM Schwartz, Joe   
 23   Alkindi Development1.22        12/26/00 6:57:16 PM  Schwartz, Joe   
 22   Alkindi Development1.21        12/26/00 5:36:59 PM  Schwartz, Joe   
 21   Alkindi Development1.20        12/20/00 5:44:56 PM  Schwartz, Joe   
 20   Alkindi Development1.19        12/19/00 3:51:58 PM  Schwartz, Joe   
 19   Alkindi Development1.18        12/18/00 6:08:24 PM  Schwartz, Joe   
 18   Alkindi Development1.17        12/18/00 4:35:38 PM  Schwartz, Joe   
 17   Alkindi Development1.16        12/18/00 12:05:39 PM Schwartz, Joe   Moved
      internal data classes to Utils package & regenerated classes from Rose.
 16   Alkindi Development1.15        12/18/00 11:24:07 AM Schwartz, Joe   
 15   Alkindi Development1.14        12/6/00 7:11:58 PM   Schwartz, Joe   Added
      catch blocks for generic exceptions and finally blocks to close all
      database connections.
 14   Alkindi Development1.13        11/30/00 12:46:43 AM Schwartz, Joe   
 13   Alkindi Development1.12        11/29/00 5:01:01 PM  Schwartz, Joe  
      Re-added dB connection close() statements.
 12   Alkindi Development1.11        11/28/00 5:51:14 PM  Schwartz, Joe   
 11   Alkindi Development1.10        11/22/00 6:49:33 PM  Schwartz, Joe   Fixed
      problems with getProductsToRate.
 10   Alkindi Development1.9         11/20/00 10:52:35 PM Schwartz, Joe   
 9    Alkindi Development1.8         11/14/00 12:53:15 PM Schwartz, Joe   
 8    Alkindi Development1.7         11/6/00 7:05:59 PM   Schwartz, Joe   
 7    Alkindi Development1.6         11/6/00 11:50:21 AM  Schwartz, Joe   
 6    Alkindi Development1.5         11/3/00 1:29:41 PM   Schwartz, Joe   Got
      rid of dependence on Parse2 and Parse3 objects. Added more error handling.
 5    Alkindi Development1.4         11/2/00 6:31:59 PM   Schwartz, Joe   
 4    Alkindi Development1.3         10/22/00 10:38:45 AM Schwartz, Joe  
      AppianDelivery 10.20.00
 3    Alkindi Development1.2         10/22/00 10:34:10 AM Schwartz, Joe   
 2    Alkindi Development1.1         10/17/00 2:22:49 PM  Schwartz, Joe  
      Delivery 10.16.00
 1    Alkindi Development1.0         10/7/00 4:34:45 PM   Schwartz, Joe   
$
$NoKeywords$
 */

/**
*	This class is a helper for RatingManager that deals with database queries.
 */
public class RatingMgrDBUtils 
{
	
	/**
	 * Fraction of user who have "seen" a Product - used in determining popular vs obscure <b>new</b> Products.
	 */
	static double propP;
	
	/**
	 * Number of user who have rated a Product - used in detmining popluar vs obscure <b>new</b> Products.
	 */
	static int propQ;
	
	/**
	 * Fraction of user who have "seen" a Product - used in determining popularity of <b>old</b> Products
	 */
	static double propPi;
	
	/**
	 * Number of user who have rated a Product - used in detmining popluar vs obscure <b>old</b> Products.
	 */
	static int propQi;
	
	/**
	 * A "fallback" function to select <b>old</b> products for a given user and product cluster. This routine returns an ArrayList of Long objects.
	 * @param user the user in question
	 * @param pcid the product cluster id
	 * @throws AlkExcept
	 * @return ArrayList
	 * @roseuid 3A4B660B0167
	 */
	public ArrayList getNewProdByPC(SystemUser user, int pcid) throws AlkExcept 
	{
		return getProdByPC(user, pcid, "{ call pkg_ALKINDI_EVALUATION.sp_SEL_NEW_PROD_DATA_BY_PC(?,?,?,?)}");
	}
	
	/**
	 * Retrieves <b>obscure new</b> Products to rate for the given user, based on statistics stored for the given Product Cluster. 
	 * The returned value is an ArrayList of Integer objects.
	 * @param user the user in question
	 * @param pcid the Product Cluster from which to select Products
	 * @return ArrayList
	 * @Exception AlkExcept
	 * @roseuid 3A4B660B00BB
	 */
	public ArrayList getNewProdObscure(SystemUser user, int pcid) throws AlkExcept 
	{
		final String mName = "getNewProdObscure";
		Connection con = null;
		ArrayList retList = new ArrayList();
		try
         {
			
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{ call pkg_alkindi_evaluation.sp_sel_new_prod_data_by_pc2(?,?,?,?,?,?)}");  
			cs.setInt (1, pcid);
			cs.setDouble (2, propP);
			cs.setInt (3, propQ);
			cs.setLong(4, user.id);
			cs.registerOutParameter(5, OracleTypes.CURSOR );
			cs.registerOutParameter(6, Types.INTEGER );	
			cs.executeQuery();
			int error = cs.getInt(6);
			if (error != 0) {
				throw new AlkExcept("", error);
			}
			ResultSet rs = ( (OracleCallableStatement)cs).getCursor(5); 
			while (rs.next()) 
			{
				retList.add(new Integer(rs.getInt(AlkConn.COL_PID)));
			}
			
			rs.close();
			cs.close();
			return retList;

		} //end try

		catch(SQLException se) 		{
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(),6004);
		}
		catch(Exception e) {
			throw new AlkExcept(e.getMessage(), 2311);
		}
		finally {
			try {	con.close();	}	catch(Exception e) {	}
		}
	}
	
	/**
	 * Retrieves Product Cluster data for the given user, based on statistics stored for <b>new</b> Products. 
	 * The data will be used to select a product to rate.
	 * The returned value is a HashMap with ProductClusters as keys and PCRatDataByProd objects as values.
	 * @param user
	 * @return HashMap
	 * @Exception AlkExcept
	 * @roseuid 3A4B660B008C
	 */
	public HashMap getNewProdPCData(SystemUser user) throws AlkExcept 
	{
		final String mName = "getNewProdPCData";
		Connection con = null;
		HashMap retMap = new HashMap();
		try
         {
			
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{ call pkg_ALKINDI_EVALUATION.sp_SEL_NEW_PROD_DATA_BY_USER(?,?,?)}");  
			cs.setLong (1, user.id);
			cs.registerOutParameter(2, OracleTypes.CURSOR );
			cs.registerOutParameter(3, Types.INTEGER );	
			cs.executeQuery();
			int error = cs.getInt(3);
			if (error != 0) {
				throw new AlkExcept("Error in Retrieving Values from RatingMgrDBUtils.GNMD1. Stored Procedure: sp_SEL_NEW_PROD_DATA_BY_USER ",error);
			}
			ResultSet rs = ( (OracleCallableStatement)cs).getCursor(2); 
			while (rs.next()) 
			{
				ProductCluster pc = new ProductCluster(rs.getInt (AlkConn.COL_PCID));
				PCRatDataByProd pd = new PCRatDataByProd();
				pd.n = rs.getDouble (AlkConn.COL_NI);
				pd.ei = rs.getDouble(AlkConn.COL_EI);
				pd.epi = rs.getDouble(AlkConn.COL_EPI);
				pd.mi = rs.getInt(AlkConn.COL_MI);
				retMap.put(pc, pd);
			}
			rs.close();
			cs.close();
			return retMap;
			
		} //END try

		catch(SQLException se) 		{
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(),6004);
		}
		catch(Exception e) {
			throw new AlkExcept(e.getMessage(), 2309);
		}
		finally {
			try {
				con.close();
			}
			catch(Exception e) {
			}
		}
	}
	
	/**
	 * Retrieves <b>popular new</b> Products to rate for the given user, based on statistics stored for the given Product Cluster.
	 * The returned value is an ArrayList of Integer objects.
	 * @param user the user in question
	 * @param pcid the Product Cluster from which to select Products
	 * @return ArrayList
	 * @Exception AlkExcept
	 * @roseuid 3A4B660B009C
	 */
	public ArrayList getNewProdPopular(SystemUser user, int pcid) throws AlkExcept 
	{
		 final String mName = "getNewProdPopular";
		Connection con = null;
		ArrayList retList = new ArrayList();
		try
         {
			
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{ call pkg_ALKINDI_EVALUATION.sp_SEL_NEW_PROD_DATA_BY_PC1(?,?,?,?,?,?)}");  
			cs.setInt (1, pcid);
			cs.setDouble (2, propP);
			cs.setInt (3, propQ);
			cs.setLong(4, user.id);
			cs.registerOutParameter(5, OracleTypes.CURSOR );
			cs.registerOutParameter(6, Types.INTEGER );	
			cs.executeQuery();
			int error = cs.getInt(6);
			if (error != 0) {
				throw new AlkExcept("", error);
			}
			ResultSet rs = ( (OracleCallableStatement)cs).getCursor(5); 
			while (rs.next()) 
			{
				retList.add(new Integer(rs.getInt(AlkConn.COL_PID)));
			}

			rs.close();
			cs.close();
			return retList;
			
		} //end try

		catch(SQLException se) 		{
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(),6004);
		}
		catch(Exception e) {
			throw new AlkExcept(e.getMessage(), 2310);
		}
		finally {
			try {
				con.close();
			}
			catch (Exception e) {
			}
		}
	}
	
	/**
	 * @param user the userin question
	 * @param pcid the id of the product cluster
	 * @throws AlkExcept
	 * @return ArrayList
	 * @roseuid 3A4B660B0186
	 */
	public ArrayList getOldProdByPC(SystemUser user, int pcid) throws AlkExcept 
	{
		return getProdByPC(user, pcid, "{ call pkg_ALKINDI_EVALUATION.sp_SEL_OLD_PROD_DATA_BY_PC(?,?,?,?)}");
	}
	
	/**
	 * Retrieves <b>obscure old</b> Products to rate for the given user, based on statistics stored for the given Product Cluster. 
	 * The returned value is an ArrayList of Integer objects.
	 * @param user the SystemUser for whom the data should be retrieved. 
	 * @param pcid the ProductCluster ID for which data should be retrieved.
	 * @param omega a tunable parameter of the Rating engine
	 * @param delta another tunable paramter of the Rating engine.
	 * @return ArrayList
	 * @Exception AlkExcept
	 * @roseuid 3A4B660B0119
	 */
	public ArrayList getOldProdObscure(SystemUser user, int pcid, int omega, int delta) throws AlkExcept 
	{
		final String mName = "getOldProdObscure";
		Connection con = null;
		ArrayList retList = new ArrayList();
		try
         {
			
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{ call pkg_ALKINDI_EVALUATION.sp_SEL_OLD_PROD_DATA_BY_PC2(?,?,?,?,?,?,?,?)}");  
			cs.setInt (1, pcid);
			cs.setDouble (2, propPi);
			cs.setInt (3, propQi);
			cs.setInt (4, omega);
			cs.setInt (5, delta);
			cs.setLong(6, user.id);
			cs.registerOutParameter(7, OracleTypes.CURSOR );
			cs.registerOutParameter(8, Types.INTEGER );	
			cs.executeQuery();
			ResultSet rs = ( (OracleCallableStatement)cs).getCursor(7); 
			int error = cs.getInt(8);
			if (error != 0) {
				throw new AlkExcept("", error);
			}
			while (rs.next()) 
			{
				retList.add(new Integer(rs.getInt (AlkConn.COL_PID)));
			}
			rs.close();
			cs.close();
			return retList;
			
		} //END try

		catch(SQLException se) {
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(),6004);
		}
		catch(Exception e) {
			throw new AlkExcept(e.getMessage(), 2314);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * Retrieves Product Cluster data for the given user, based on statistics stored for <b>old</b> products. 
	 * The data will be used to select a product to rate.
	 * The returned value is a HashMap with ProductClusters as keys and PCRatDataByProd objects as values.
	 * @param user, pcid
	 * @return ArrayList
	 * @Exception AlkExcept
	 * @roseuid 3A4B660B00DA
	 */
	public HashMap getOldProdPCData(SystemUser user) throws AlkExcept 
	{
		final String mName = "getOldProdPCData";
		Connection con = null;
		HashMap retMap = new HashMap();
		try
         {
			
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{ call pkg_ALKINDI_EVALUATION.sp_SEL_OLD_PROD_DATA_BY_USER(?,?,?)}");  
			cs.setLong (1, user.id);
			cs.registerOutParameter(2, OracleTypes.CURSOR );
			cs.registerOutParameter(3, Types.INTEGER );	
			cs.executeQuery();
			int error = cs.getInt(3);
			if (error != 0) {
				throw new AlkExcept("", error);
			}
			ResultSet rs = ( (OracleCallableStatement)cs).getCursor(2);
			while (rs.next()) 
			{
				ProductCluster pc = new ProductCluster(rs.getInt (AlkConn.COL_PCID));
				PCRatDataByProd pd = new PCRatDataByProd();
				pd.n = rs.getDouble (AlkConn.COL_NI);
				pd.ei = rs.getDouble(AlkConn.COL_EI);
				pd.epi = rs.getDouble(AlkConn.COL_EPI);
				pd.mi = rs.getInt(AlkConn.COL_MI);
				retMap.put(pc, pd);
			}

			rs.close();
			cs.close();
			return retMap;			
		} //END try

		catch(SQLException se) 		{
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(),6004);
		}
		catch(Exception e) {
			throw new AlkExcept(e.getMessage(), 2312);
		}
		finally{
			try { con.close();	}	catch(Exception e) {	}
		}
	}
	
	/**
	 * Retrieves <b>popular old</b> Products to rate for the given user, based on statistics stored for the given Product Cluster. 
	 * The returned value is an ArrayList of Integer objects.
	 * @param user the user in question
	 * @param pcID the product cluster from which to select products
	 * @return ArrayList
	 * @Exception AlkExcept
	 * @roseuid 3A4B660B00EA
	 */
	public ArrayList getOldProdPopular(SystemUser user, int pcID, int omega, int delta) throws AlkExcept 
	{
		final String mName = "getOldProdPopular";
		Connection con = null;
		ArrayList retList = new ArrayList();
		try
         {
			
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{ call pkg_ALKINDI_EVALUATION.sp_SEL_OLD_PROD_DATA_BY_PC1(?,?,?,?,?,?,?,?)}");  
			cs.setInt (1, pcID);
			cs.setDouble (2, propPi);
			cs.setInt (3, propQi);
			cs.setInt (4, omega);
			cs.setInt (5, delta);
			cs.setLong(6, user.id);
			cs.registerOutParameter(7, OracleTypes.CURSOR );
			cs.registerOutParameter(8, Types.INTEGER );	
			cs.executeQuery();
			int error = cs.getInt(8);
			if (error != 0) {
				throw new AlkExcept("", error);
			}
			ResultSet rs = ( (OracleCallableStatement)cs).getCursor(7); 
			while (rs.next()) 
			{
				retList.add(new Integer(rs.getInt (AlkConn.COL_PID)));
			}

			rs.close();
			cs.close();
			
		} //END try

		catch(SQLException se) 	{
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(),6004);
		}
		catch(Exception e) {
			throw new AlkExcept(e.getMessage(), 2313);
		}
		finally {
			try {
				con.close();
			}
			catch(Exception e) {
			}

		}
		return retList;
	}
	
	/**
	 * @roseuid 3A4B660B01A5
	 */
	protected ArrayList getProdByPC(SystemUser user, int pcid, String storedProc) throws AlkExcept 
	{
		final String mName = "getProdByPC";
		ArrayList prodIDList = new ArrayList();
		Connection con = null;
		try
         {
			
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall(storedProc);  
			cs.setInt (1, pcid);
			cs.setLong(2, user.id);
			cs.registerOutParameter(3, OracleTypes.CURSOR );
			cs.registerOutParameter(4, Types.INTEGER );	
			cs.executeQuery();
			int error = cs.getInt(4);
			if (error != 0) {
				throw new AlkExcept("", error);
			}
			ResultSet rs = ( (OracleCallableStatement)cs).getCursor(3); 
			while (rs.next()) 
			{
				prodIDList.add(new Integer(rs.getInt (AlkConn.COL_PID)));
			}

			rs.close();
			cs.close();
			
		} //END try

		catch(SQLException se) 	{
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(),6004);
		}
		catch(Exception e) {
			throw new AlkExcept(e.getMessage(), 2316);
		}
		finally {
			try {
				con.close();
			}
			catch(Exception e) {
			}
		}
		return prodIDList;
	}
	
	/**
	 * Returns information about the given User's evaluations.
	 * Calls the Stored Procedures <b>pkg_ALKINDI_EVALUATION.GETUSERSTATS</b> and <b>pkg_ALKINDI_EVALUATION.GETEVERYONESTATS</b>.
	 * @param user The User whose evaluation information is to be queried.
	 * @throws AlkExcept
	 * @roseuid 3AF6F6BB00AB
	 */
	public UserEvalStats getUserEvalStats(final SystemUser user) throws AlkExcept 
	{
		final String mName = "getUserEvalStats";
		Connection con = null;
		
		UserEvalStats ues = null;
		try {
			con = AlkConn.getConnection();
			
			CallableStatement csUser =
			con.prepareCall("{ call pkg_ALKINDI_EVALUATION.GETUSERSTATS(?,?,?,?,?,?,?) }");
			csUser.setLong(1, user.id);
			csUser.registerOutParameter(2, Types.INTEGER);
			csUser.registerOutParameter(3, Types.INTEGER);
			csUser.registerOutParameter(4, Types.FLOAT);
			csUser.registerOutParameter(5, Types.INTEGER);
			csUser.registerOutParameter(6, Types.INTEGER);
			csUser.registerOutParameter(7, Types.INTEGER);
			csUser.executeQuery();
			
			CallableStatement csEveryone =
			con.prepareCall("{ call pkg_ALKINDI_EVALUATION.GETEVERYONESTATS(?,?,?,?,?,?) }");
			csEveryone.registerOutParameter(1, Types.INTEGER);
			csEveryone.registerOutParameter(2, Types.INTEGER);			
			csEveryone.registerOutParameter(3, Types.FLOAT);	
			csEveryone.registerOutParameter(4, Types.INTEGER);		
			csEveryone.registerOutParameter(5, Types.INTEGER);
			csEveryone.registerOutParameter(6, Types.INTEGER);			
			csEveryone.executeQuery();
			
			ues = new UserEvalStats(
				user,
				csUser.getInt(5), 		/* core shown */
				csUser.getInt(6),		/* selectable shown */
				csEveryone.getInt(4),	/* total core */
				csEveryone.getInt(5),	/* total selectable */
				csUser.getFloat(4),		/* user Epsilon */
				csEveryone.getFloat(3)	/* overall Epsilon */
			);
			
			csEveryone.close();
			csUser.close();
			
		}
		catch(SQLException se) 	{
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(),6004);
		}	
		catch (Exception e) {
			throw new AlkExcept("Unexpected error in " + mName + ": " +
			e.getMessage(), 2318, e);
		}
		finally {
			try { 	con.close(); } catch (Exception e) {}
		}
		return ues;
	}
	
	/**
	 * Retrieves Product Cluster data based on statistics stored for the given user. 
	 * The data will be used to select a product cluster from which to get products to rate.
	 * The returned value is a HashMap with ProductClusters as keys and PCRatDataByUser objects as values.
	 * @param user
	 * @return HashMap
	 * @Exception AlkExcept
	 * @roseuid 3A4B660B005E
	 */
	public HashMap getUserPCData(SystemUser user) throws AlkExcept 
	{
		final String mName = "getUserPCData";
		HashMap retMap = new HashMap();
		Connection con = null;
		try
         {
			
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{ call pkg_ALKINDI_EVALUATION.sp_SEL_USER_DATA(?,?,?)}");  
			cs.setLong (1, user.id);
			cs.registerOutParameter(2, OracleTypes.CURSOR );
			cs.registerOutParameter(3, Types.INTEGER );	
			cs.executeQuery();
			int error = cs.getInt(3);
			if (error != 0) {
				throw new AlkExcept("", error);
			}
			
			ResultSet rs = ( (OracleCallableStatement)cs).getCursor(2); 
				
			while (rs.next()) 
			{
				ProductCluster pc = new ProductCluster(rs.getInt (AlkConn.COL_PCID));
				PCRatDataByUser ud = new PCRatDataByUser();
				ud.b = rs.getDouble (AlkConn.COL_BI);
				ud.E = rs.getDouble (AlkConn.COL_EI);
				ud.R = rs.getDouble (AlkConn.COL_RI);
				ud.F = rs.getDouble (AlkConn.COL_FI);
				ud.Eave = rs.getDouble (AlkConn.COL_EAVE);
				ud.Rave = rs.getDouble (AlkConn.COL_RAVE);
				ud.n = rs.getDouble (AlkConn.COL_NI);
				retMap.put(pc, ud);
			}
			
			rs.close();
			cs.close();
			return retMap;

		} //END try

		catch(SQLException se) 		{
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(),6004);
		}
		catch(Exception e) {
			throw new AlkExcept(e.getMessage(), 2307);
		}
		finally{
			try {
				con.close();
			}
			catch(Exception e) {
			}
		}
	}
	
	/**
	 * Retrieves Products to rate for a given Product Cluster, based on statistics stored for the given user.
	 * The data will be used to select a product to present for rating.
	 * The returned value is an ArrayList of Integer objects representing product IDs.
	 * @param user the User in question
	 * @param pcID the ProductCluster ID
	 * @return HashMap
	 * @Exception AlkExcept
	 * @roseuid 3A4B660B006E
	 */
	public ArrayList getUserProdData(SystemUser user, int pcID) throws AlkExcept 
	{
		final String mName = "getUserProdData";
		Connection con = null;
		ArrayList retList = new ArrayList();
		try
         {
			
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{ call pkg_ALKINDI_EVALUATION.sp_SEL_PROD_BY_USER_DATA(?,?,?,?)}");  
			cs.setInt (1, pcID);
			cs.setLong (2, user.id);
			cs.registerOutParameter(3, OracleTypes.CURSOR );
			cs.registerOutParameter(4, Types.INTEGER );	
			cs.executeQuery();
			int error = cs.getInt(4);
			if (error != 0) {
				throw new AlkExcept("", error);
			}
			ResultSet rs = ( (OracleCallableStatement)cs).getCursor(3); 
			while (rs.next()) 
			{
				retList.add(new Integer(rs.getInt(AlkConn.COL_PID)));
			}
		
			rs.close();
			cs.close();
			return retList;

		} //END try

		catch(SQLException se) {
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(),6004);
		}
		catch(Exception e) {
			throw new AlkExcept(e.getMessage(), 2308);
		}
		finally {	try {	con.close();}		catch (Exception e) {}
		}
	}
	
	/**
	 * Retrieves a user's ratings from the database.
	 * @param user the user in question.
	 * @return RatingList
	 * @throws AlkExcept
	 * @roseuid 3AF6F622000F
	 */
	public Alkindi.Data.RatingList getUserRatings(Alkindi.Data.SystemUser user) throws AlkExcept 
	{
		final String mName = "getUserRatings";
		String errmsg = "";
		RatingList rl = new RatingList();
		Connection con = null;

		try{
			
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall(
				"{ call pkg_ALKINDI_EVALUATION.sp_SEL_ProdEval_By_User(?,?,?) }");
			cs.setLong(1,user.id);
			cs.registerOutParameter(2,OracleTypes.CURSOR);
			cs.registerOutParameter(3,Types.INTEGER);
			cs.executeQuery();

			ResultSet rs = (ResultSet) cs.getObject(2);
			int error = cs.getInt(3);

			while (rs.next()) {
				rl.add(new Rating(user, new Product(rs.getInt(AlkConn.COL_PID)),rs.getInt(AlkConn.COL_EVALSCALEID)));
			}

			rs.close();
			cs.close();
		}
		catch(SQLException se) {
			errmsg = "SQL Exception: " + se.getMessage();
			throw new AlkExcept(errmsg + " in " + mName, 6004);
		}
		catch(Exception e) {
			errmsg = "Unexpected error: " + e.getMessage();
			throw new AlkExcept(errmsg + " in " + mName, 2303);
		}
		finally {
			try {
				con.close();
			}
			catch (Exception e) {
			}
		}
		return rl;
	}
	
	/**
	 * Constructor to read Engine properties.
	 * @roseuid 3A4B660B005D
	 */
	public RatingMgrDBUtils() throws AlkExcept 
	{
		propP = PropertyManager.getDouble("P");
		propQ = PropertyManager.getInt("Q");
		propPi = PropertyManager.getDouble("Pi");
		propQi = PropertyManager.getInt("Qi");
	}
}
